/* global jsyaml LZString */
"use strict";

// ---------- Configuration ----------
const KINKS_YAML_URL = "yaml/kinks.yml";
const INTEREST_YAML_URL = "yaml/interest_scale.yml";
const CATEGORIES_YAML_URL = "yaml/kink_categories.yml";

const URL_FRAGMENT_KEY = "k";
const AGE_SESSION_KEY = "age_verified_18plus";

// Share-link session state (kept only for the lifetime of the tab).
// This allows users to fix mistakes and re-share without generating new links.
const SHARE_SESSION_KEY = "active_share";

// If you later add a Worker, expose these endpoints from it.
// For now, the UI will gracefully fall back to URL-fragment sharing when
// the endpoints are not reachable.
const SHARE_API_BASE = "https://api.sharekinks.com/api/share";

// If you keep running into caching during dev, flip this to true
const DEV_NO_CACHE = true;

// ---------- State ----------
let kinkData = [];
let interestScale = [];
let categories = [];

const interestById = new Map();
const categoryById = new Map();

let selectedCategoryIds = new Set(); // empty = All
let selectedInterestIds = new Set(); // empty = All
let sortMode = "name_asc";

const interestRankById = new Map(); // filled from YAML rank

let selectedOnlyFilterEnabled = false;
let showKinkDescriptions = true;

// ---------- Helpers ----------
function alphaByName(a, b) {
  const an = (a?.name || "").toString();
  const bn = (b?.name || "").toString();
  return an.localeCompare(bn, undefined, { sensitivity: "base" });
}

function isDevMode() {
  const params = new URLSearchParams(location.search);
  return (
    params.has("dev") ||
    location.hostname === "localhost" ||
    location.hostname === "127.0.0.1"
  );
}

function devFetchOptions() {
  if (!DEV_NO_CACHE) return {};
  if (!isDevMode()) return {};
  return { cache: "no-store" };
}

// ---------- Age Gate ----------
function setDobMaxToday() {
  const dobInput = document.getElementById("dob");
  if (!dobInput) return;
  const today = new Date();
  const yyyy = today.getFullYear();
  const mm = String(today.getMonth() + 1).padStart(2, "0");
  const dd = String(today.getDate()).padStart(2, "0");
  dobInput.max = `${yyyy}-${mm}-${dd}`;
}

function isOver18(dobStr) {
  const dob = new Date(dobStr);
  if (Number.isNaN(dob.getTime())) return false;

  const today = new Date();
  const cutoff = new Date(
    today.getFullYear() - 18,
    today.getMonth(),
    today.getDate()
  );
  return dob <= cutoff;
}

function showAgeGate() {
  document.getElementById("age-gate")?.classList.remove("d-none");
  document.getElementById("app-root")?.classList.add("d-none");
}

function showApp() {
  document.getElementById("age-gate")?.classList.add("d-none");
  document.getElementById("app-root")?.classList.remove("d-none");
}

function handleAgeGateSubmit(event) {
  event.preventDefault();

  const dobInput = document.getElementById("dob");
  const errorEl = document.getElementById("age-error");
  if (!dobInput || !errorEl) return;

  const dobStr = dobInput.value;

  if (!dobStr) {
    errorEl.textContent = "Please enter your date of birth.";
    errorEl.classList.remove("d-none");
    return;
  }

  if (!isOver18(dobStr)) {
    errorEl.textContent = "You must be at least 18 years old to use this tool.";
    errorEl.classList.remove("d-none");
    return;
  }

  errorEl.classList.add("d-none");
  sessionStorage.setItem(AGE_SESSION_KEY, "true");

  showApp();
  void initApp();
}

function resetAgeCheck() {
  sessionStorage.removeItem(AGE_SESSION_KEY);
  showAgeGate();
}

// ---------- URL Hash State ----------
let liveUpdateTimeout = null;

function scheduleLiveHashUpdate(delayMs = 400) {
  if (liveUpdateTimeout) {
    clearTimeout(liveUpdateTimeout);
  }

  liveUpdateTimeout = setTimeout(() => {
    writeHashFromUI();
    indicateStateChanged();
    liveUpdateTimeout = null;
  }, delayMs);
}

let sortUpdateTimeout = null;

function scheduleSortUpdate(delayMs = 500) {
  // Only needed when sort depends on interest
  if (sortMode === "name_asc") return;

  if (sortUpdateTimeout) {
    clearTimeout(sortUpdateTimeout);
  }

  sortUpdateTimeout = setTimeout(() => {
    sortCards();
    applyFiltersToCards();
    sortUpdateTimeout = null;
  }, delayMs);
}

function writeHashFromUI() {
  // When using a short share link (?s= / ?m=) or an active session share, we
  // intentionally keep the address bar "safe" and stable (no long fragments).
  // In that mode, the Share… button updates the stored share state instead.
  const params = new URLSearchParams(window.location.search);
  const inShareMode = params.has("s") || params.has("m") || Boolean(loadActiveShare());
  if (inShareMode) return;

  const state = readStateFromUI();
  const newHash = encodeStateToHash(state);
  history.replaceState(null, "", newHash);
}

function encodeStateToHash(state) {
  const json = JSON.stringify(state);

  // Prefer the new compact format when available.
  // Falls back to the legacy format if LZString is not present.
  if (typeof LZString !== "undefined" && LZString?.compressToEncodedURIComponent) {
    const encoded = LZString.compressToEncodedURIComponent(json);
    return `#${URL_FRAGMENT_KEY}=lz:${encoded}`;
  }

  // Legacy: Base64(encodeURIComponent(JSON))
  const utf8 = encodeURIComponent(json);
  const base64 = btoa(utf8);
  return `#${URL_FRAGMENT_KEY}=${base64}`;
}

function decodeStateFromHash(hash) {
  if (!hash || !hash.startsWith(`#${URL_FRAGMENT_KEY}=`)) return null;

  const payload = hash.slice(`#${URL_FRAGMENT_KEY}=`.length);

  // New: lz:<encoded>
  if (payload.startsWith("lz:")) {
    if (typeof LZString === "undefined" || !LZString?.decompressFromEncodedURIComponent) {
      console.error("LZString missing; cannot decode lz: hash format.");
      return null;
    }

    try {
      const encoded = payload.slice(3);
      const json = LZString.decompressFromEncodedURIComponent(encoded);
      if (typeof json !== "string" || !json) {
        throw new Error("LZString decompression returned empty/invalid output.");
      }
      return JSON.parse(json);
    } catch (err) {
      console.error("Failed to decode LZString state from hash:", err);
      return null;
    }
  }

  // Legacy: Base64(encodeURIComponent(JSON))
  try {
    const utf8 = atob(payload);
    const json = decodeURIComponent(utf8);
    return JSON.parse(json);
  } catch (err) {
    console.error("Failed to decode legacy state from hash:", err);
    return null;
  }
}


  // ---------- multi-column header equalization ----------
let equalizeTimeout = null;

function isMultiColumnGrid() {
  const row = document.getElementById("kink-row");
  if (!row) return false;

  const cols = Array.from(row.querySelectorAll(":scope > .col")).filter(
    (c) => !c.classList.contains("d-none")
  );

  if (cols.length < 2) return false;

  // If any two adjacent visible cols share the same top, we have multiple columns.
  for (let i = 1; i < cols.length; i += 1) {
    if (cols[i].offsetTop === cols[i - 1].offsetTop) return true;
  }

  return false;
}

function scheduleEqualizeHeaders(delayMs = 80) {
  if (equalizeTimeout) clearTimeout(equalizeTimeout);
  equalizeTimeout = setTimeout(() => {
    equalizeTimeout = null;
    equalizeHeaderHeights();
    equalizeRoleMetaHeights();

  }, delayMs);
}

function applyDescriptionVisibility() {
  document.body.classList.toggle(
    "hide-kink-descriptions",
    !showKinkDescriptions
  );
  scheduleEqualizeHeaders(50);
}


function equalizeHeaderHeights() {
  if (!isMultiColumnGrid()) {
    document.querySelectorAll(".kink-header").forEach((h) => {
      h.style.minHeight = "";
    });
    return;
  }

  const row = document.getElementById("kink-row");
  if (!row) return;

  const cols = Array.from(row.querySelectorAll(":scope > .col")).filter(
    (c) => !c.classList.contains("d-none")
  );

  cols.forEach((col) => {
    const header = col.querySelector(".kink-header");
    if (header) header.style.minHeight = "";
  });

  const groups = new Map();
  for (const col of cols) {
    const top = col.offsetTop;
    if (!groups.has(top)) groups.set(top, []);
    groups.get(top).push(col);
  }

  for (const group of groups.values()) {
    let max = 0;
    const headers = [];

    for (const col of group) {
      const header = col.querySelector(".kink-header");
      if (!header) continue;
      headers.push(header);
      max = Math.max(max, header.getBoundingClientRect().height);
    }

    for (const header of headers) {
      header.style.minHeight = `${Math.ceil(max)}px`;
    }
  }
}

function equalizeRoleMetaHeights() {
  if (!isMultiColumnGrid()) {
    document.querySelectorAll(".kink-role-meta").forEach((el) => {
      el.style.minHeight = "";
    });
    return;
  }

  const row = document.getElementById("kink-row");
  if (!row) return;

  const cols = Array.from(row.querySelectorAll(":scope > .col")).filter(
    (c) => !c.classList.contains("d-none")
  );

  // Reset before measuring
  for (const col of cols) {
    col.querySelectorAll(".kink-role-meta").forEach((m) => {
      m.style.minHeight = "";
    });
  }

  // Group cards by visual grid row using offsetTop
  const groupsByTop = new Map();
  for (const col of cols) {
    const top = col.offsetTop;
    if (!groupsByTop.has(top)) groupsByTop.set(top, []);
    groupsByTop.get(top).push(col);
  }

  for (const groupCols of groupsByTop.values()) {
    // roleIndex -> max meta height in this grid row
    const maxByRoleIndex = new Map();

    // Pass 1: compute maxima
    for (const col of groupCols) {
      const roleRows = col.querySelectorAll(".role-row[data-role-index]");
      for (const rr of roleRows) {
        const idx = rr.dataset.roleIndex;
        const meta = rr.querySelector(".kink-role-meta");
        if (idx == null || !meta) continue;

        const h = meta.getBoundingClientRect().height;
        const prev = maxByRoleIndex.get(idx);
        if (prev === undefined || h > prev) {
          maxByRoleIndex.set(idx, h);
        }
      }
    }

    // Pass 2: apply minHeight
    for (const col of groupCols) {
      const roleRows = col.querySelectorAll(".role-row[data-role-index]");
      for (const rr of roleRows) {
        const idx = rr.dataset.roleIndex;
        const meta = rr.querySelector(".kink-role-meta");
        if (idx == null || !meta) continue;

        const target = maxByRoleIndex.get(idx);
        if (target === undefined) continue;

        meta.style.minHeight = `${Math.ceil(target)}px`;
      }
    }
  }
}


function maybeUpgradeLegacyHashToLz() {
  // Only upgrade if:
  // 1) We have a legacy hash present
  // 2) LZString is available
  // 3) We can successfully decode the legacy payload
  // 4) The upgraded hash would actually be different
  const hash = location.hash || "";
  if (!hash.startsWith(`#${URL_FRAGMENT_KEY}=`)) return;

  const payload = hash.slice(`#${URL_FRAGMENT_KEY}=`.length);
  if (!payload || payload.startsWith("lz:")) return;

  if (typeof LZString === "undefined" || !LZString?.compressToEncodedURIComponent) {
    return;
  }

  const state = decodeStateFromHash(hash);
  if (!state) return;

  const upgraded = encodeStateToHash(state);
  if (upgraded !== hash) {
    history.replaceState(null, "", upgraded);
  }
}

// ---------- YAML Loading ----------
async function loadYaml(url) {
  const resp = await fetch(url, devFetchOptions());
  if (!resp.ok) throw new Error(`Failed to load ${url}: ${resp.status}`);
  const text = await resp.text();
  return jsyaml.load(text);
}

const kinkById = new Map();

function rebuildKinkIndex() {
  kinkById.clear();
  for (const kink of kinkData) {
    if (kink?.id) kinkById.set(kink.id, kink);
  }
}

async function initApp() {
  const loadingEl = document.getElementById("loading");

  try {
    const [kinksYaml, interestYaml, categoriesYaml] = await Promise.all([
      loadYaml(KINKS_YAML_URL),
      loadYaml(INTEREST_YAML_URL),
      loadYaml(CATEGORIES_YAML_URL),
    ]);

    kinkData = Array.isArray(kinksYaml?.kinks) ? kinksYaml.kinks : [];
    rebuildKinkIndex();
    interestScale = Array.isArray(interestYaml?.interest_scale)
      ? interestYaml.interest_scale
      : [];
    interestRankById.clear();
    interestScale.forEach((i) =>
      interestRankById.set(i.id, Number(i.rank ?? 0))
    );
    categories = Array.isArray(categoriesYaml?.categories)
      ? categoriesYaml.categories
      : [];

    // Alphabetize
    kinkData.sort(alphaByName);
    categories.sort(alphaByName);

    interestById.clear();
    categoryById.clear();

    interestScale.forEach((i) => interestById.set(i.id, i));
    categories.forEach((c) => categoryById.set(c.id, c));

    if (!kinkData.length || !interestScale.length) {
      if (loadingEl)
        loadingEl.textContent =
          "No kinks or interest scale entries found in the YAML files.";
      return;
    }

    renderCategoryFilters();
    renderInterestFilters();
    renderKinks();
    await applyStateFromUrl();
    applyFiltersToCards();
    sortCards();
    initPillAutoFit();
    initPillAutoFitResizeFallback();

    loadingEl?.classList.add("d-none");
    document.getElementById("kink-container")?.classList.remove("d-none");
  } catch (err) {
    console.error(err);
    if (loadingEl)
      loadingEl.textContent =
        "Error loading YAML data. See console for details.";
  }
}

function withScrollAnchor(work) {
  // Pick a point slightly below the top so we don't hit the navbar area.
  const x = Math.floor(window.innerWidth / 2);
  const y = 120;

  const beforeEl = document.elementFromPoint(x, y)?.closest(".col, .kink-card");
  const beforeTop = beforeEl ? beforeEl.getBoundingClientRect().top : null;

  work();

  // After DOM changes, restore perceived position.
  if (!beforeEl || beforeTop === null) return;

  // elementFromPoint may return a different node after reflow; re-find by dataset if possible
  const kinkId =
    beforeEl.classList.contains("kink-card")
      ? beforeEl.dataset.kinkId
      : beforeEl.querySelector(".kink-card")?.dataset.kinkId;

  const afterEl = kinkId
    ? document.querySelector(`.kink-card[data-kink-id="${kinkId}"]`)?.closest(".col, .kink-card")
    : beforeEl;

  if (!afterEl) return;

  const afterTop = afterEl.getBoundingClientRect().top;
  const delta = afterTop - beforeTop;

  if (Math.abs(delta) > 1) {
    window.scrollBy({ top: delta, left: 0, behavior: "auto" });
  }
}


// ---------- Category Filters ----------
function renderCategoryFilters() {
  const section = document.getElementById("category-filters");
  const container = document.getElementById("category-filter-buttons");
  if (!section || !container) return;

  container.innerHTML = "";

  if (!categories.length) {
    section.classList.add("d-none");
    return;
  }

  section.classList.remove("d-none");

  // "All" button clears category filters (empty set = All)
  const all_button = document.createElement("button");
  all_button.type = "button";
  all_button.className = "btn btn-sm btn-outline-primary";
  all_button.dataset.categoryId = "";
  all_button.textContent = "All";
  all_button.addEventListener("click", () => {
    selectedCategoryIds.clear();
    syncCategoryButtonsToState();
    applyFiltersToCards();
    scheduleSortUpdate(500);

    if (liveUpdateTimeout) {
      clearTimeout(liveUpdateTimeout);
      liveUpdateTimeout = null;
    }
    writeHashFromUI();
    indicateStateChanged();
  });
  container.appendChild(all_button);

  // Category buttons toggle on/off (multi-select)
  categories.forEach((cat) => {
    const btn = document.createElement("button");
    btn.type = "button";
    btn.className = "btn btn-sm btn-outline-primary";
    btn.dataset.categoryId = cat.id;
    btn.textContent = cat.name;
    btn.title = cat.description || "";
    btn.addEventListener("click", () => {
      if (selectedCategoryIds.has(cat.id)) {
        selectedCategoryIds.delete(cat.id);
      } else {
        selectedCategoryIds.add(cat.id);
      }

      syncCategoryButtonsToState();
      applyFiltersToCards();
      scheduleSortUpdate(500);

      if (liveUpdateTimeout) {
        clearTimeout(liveUpdateTimeout);
        liveUpdateTimeout = null;
      }
      writeHashFromUI();
      indicateStateChanged();
    });
    container.appendChild(btn);
  });

  // Ensure UI matches current state (e.g., after restoring from URL)
  syncCategoryButtonsToState();
}

// ---------- Search filter -------------

function tokenize(q) {
  return (q || "")
    .toLowerCase()
    .trim()
    .split(/\s+/)
    .filter(Boolean);
}

let kinkSearchQuery = "";

const kinkSearchInput = document.getElementById("kinkSearch");

if (kinkSearchInput) {
  kinkSearchInput.addEventListener("input", () => {
    kinkSearchQuery = kinkSearchInput.value.trim().toLowerCase();
    applyFiltersToCards();
  });
}

function kinkMatchesSearch(kinkId) {
  const tokens = tokenize(kinkSearchQuery);
  if (tokens.length === 0) return true;

  const kink = kinkById.get(kinkId);
  if (!kink) return false;

  const parts = [];
  parts.push(kink.name, kink.description);

  if (Array.isArray(kink.notes)) parts.push(...kink.notes);
  else parts.push(kink.notes);

  if (Array.isArray(kink.roles)) {
    for (const role of kink.roles) {
      parts.push(role?.name, role?.description);
    }
  }

  const haystack = parts.filter(Boolean).join(" ").toLowerCase();

  // AND semantics: every token must match somewhere
  return tokens.every((t) => haystack.includes(t));
}

// ---------- Pill font auto-fit (no wrapping) ----------

const PILL_FONT_MIN_PX = 11; // choose your floor
const PILL_FONT_MAX_PX = 14; // choose your ceiling (Bootstrap btn-sm-ish)
const PILL_FONT_STEP_PX = 0.25;

// Only for the interest filter row: prefer reducing gap over shrinking below min.
const FILTER_GAP_CLASSES = ["gap-2", "gap-1"];


let pillFitRaf = null;
let pillResizeObserver = null;
const lastWidths = new WeakMap();

/**
 * Returns true if all direct/descendant .interest-pill buttons inside container
 * are on a single visual line (no wrapping).
 */
function pillsAreSingleLine(container) {
  const pills = container.querySelectorAll(".interest-pill");
  if (pills.length <= 1) return true;

  // Use offsetTop: if any pill has a different offsetTop, we wrapped.
  const firstTop = pills[0].offsetTop;
  for (let i = 1; i < pills.length; i += 1) {
    if (pills[i].offsetTop !== firstTop) return false;
  }
  return true;
}

function setPillFontPx(container, px) {
  container.style.setProperty("--pill-font-size", `${px}px`);
}


function getCurrentGapIndex(container) {
  for (let i = 0; i < FILTER_GAP_CLASSES.length; i += 1) {
    if (container.classList.contains(FILTER_GAP_CLASSES[i])) return i;
  }
  return -1;
}

/**
 * Binary search the max font size that keeps pills on one line.
 * Writes the result to --pill-font-size on the container.
 */
function fitPillFont(container) {
  if (container.offsetParent === null) return;
  const w = container.clientWidth;
  if (lastWidths.get(container) === w) return;
  lastWidths.set(container, w);

  const isFilter = isInterestFilterContainer(container);

  // For the filter row, we may adjust gap if needed.
  // Start from current gap (whatever is in the DOM).
  let gapIdx = isFilter ? getCurrentGapIndex(container) : -1;

  // We'll try: (gap stays) + binary-search font. If still wraps at min, reduce gap and retry.
  const tryFitWithCurrentGap = () => {
    // Fits at max? Great.
    setPillFontPx(container, PILL_FONT_MAX_PX);
    if (pillsAreSingleLine(container)) return true;

    // If it doesn't fit even at min, we'll return false (caller may reduce gap).
    setPillFontPx(container, PILL_FONT_MIN_PX);
    if (!pillsAreSingleLine(container)) return false;

    // Binary search between min and max (ticks).
    const ticks = Math.round(
      (PILL_FONT_MAX_PX - PILL_FONT_MIN_PX) / PILL_FONT_STEP_PX
    );
    let loTick = 0;
    let hiTick = ticks;

    while (loTick < hiTick) {
      const midTick = Math.ceil((loTick + hiTick) / 2);
      const mid = PILL_FONT_MIN_PX + midTick * PILL_FONT_STEP_PX;

      setPillFontPx(container, mid);

      if (pillsAreSingleLine(container)) {
        loTick = midTick;
      } else {
        hiTick = midTick - 1;
      }
    }

    const best = PILL_FONT_MIN_PX + loTick * PILL_FONT_STEP_PX;
    setPillFontPx(container, best);

    // After choosing best, verify we actually fit (should, but be safe).
    return pillsAreSingleLine(container);
  };

  // Non-filter rows: just do the normal fit.
  if (!isFilter) {
    void tryFitWithCurrentGap();
    return;
  }

  // Filter row: try current gap first.
  if (tryFitWithCurrentGap()) return;

  // If still wrapping at min font, reduce gap progressively and retry.
  // If no gap-* class was present, treat that as "gap-2" baseline.
  if (gapIdx === -1) gapIdx = 0;

  while (gapIdx < FILTER_GAP_CLASSES.length - 1) {
    gapIdx += 1;
    setGapByIndex(container, gapIdx);

    if (tryFitWithCurrentGap()) return;
  }

  // At smallest gap and still wrapping: leave at gap-0 + min font.
  setGapByIndex(container, FILTER_GAP_CLASSES.length - 1);
  setPillFontPx(container, PILL_FONT_MIN_PX);
}


function setGapByIndex(container, idx) {
  FILTER_GAP_CLASSES.forEach((cls) => container.classList.remove(cls));
  if (idx >= 0 && idx < FILTER_GAP_CLASSES.length) {
    container.classList.add(FILTER_GAP_CLASSES[idx]);
  }
}

function isInterestFilterContainer(container) {
  return container.id === "interest-filter-buttons";
}

function fitAllPillContainers() {
  // Fit both the interest filter pills and the per-role pill rows
  document.querySelectorAll(".interest-pills").forEach((container) => {
    // If container (or ancestors) are hidden, measurements are unreliable.
    // Skip hidden; we’ll fit when it becomes visible (on next call).
    if (container.offsetParent === null) return;
    fitPillFont(container);
  });
}

function scheduleFitAllPillContainers() {
  if (pillFitRaf) cancelAnimationFrame(pillFitRaf);
  pillFitRaf = requestAnimationFrame(() => {
    pillFitRaf = null;
    fitAllPillContainers();
  });
}

function initPillAutoFit() {
  // Refit on any container size changes (cards change width at breakpoints)
  if (pillResizeObserver) pillResizeObserver.disconnect();

  pillResizeObserver = new ResizeObserver(() => {
    scheduleFitAllPillContainers();
  });

  // Observe each pill container
  document.querySelectorAll(".interest-pills").forEach((container) => {
    pillResizeObserver.observe(container);
  });

  // Initial fit
  scheduleFitAllPillContainers();
}

function initPillAutoFitResizeFallback() {
  let resizeTimeout = null;

  window.addEventListener("resize", () => {
    if (resizeTimeout) {
      clearTimeout(resizeTimeout);
    }

    resizeTimeout = setTimeout(() => {
      resizeTimeout = null;
      scheduleFitAllPillContainers();
    }, 100);
  });
}


// ---------- Interest Filters ----------

function renderInterestFilters() {
  const section = document.getElementById("interest-filters");
  const container = document.getElementById("interest-filter-buttons");
  if (!section || !container) return;

  container.innerHTML = "";

  if (!interestScale.length) {
    section.classList.add("d-none");
    return;
  }
  section.classList.remove("d-none");

  // All button clears interest filters
  const all_btn = document.createElement("button");
  all_btn.type = "button";
  all_btn.className = "btn btn-sm btn-outline-primary interest-pill";
  all_btn.dataset.interestId = "";
  all_btn.textContent = "All";
  all_btn.addEventListener("click", () => {
    selectedInterestIds.clear();
    syncInterestButtonsToState();
    applyFiltersToCards();
    scheduleSortUpdate(500);

    if (liveUpdateTimeout) {
      clearTimeout(liveUpdateTimeout);
      liveUpdateTimeout = null;
    }
    writeHashFromUI();
    indicateStateChanged();
  });
  container.appendChild(all_btn);

  // One button per interest item (multi-select)
  interestScale.forEach((interest) => {
    const btn = document.createElement("button");
    btn.type = "button";

    const defCls = interest.btn_class || "btn-outline-secondary";
    const actCls = interest.btn_active_class || "btn-secondary";

    btn.className = `btn btn-sm rounded-pill interest-pill ${defCls}`;
    btn.dataset.interestId = interest.id;
    btn.dataset.defaultClass = defCls;
    btn.dataset.activeClass = actCls;
    btn.textContent = interest.label;
    btn.title = interest.description || "";

    btn.addEventListener("click", () => {
      if (selectedInterestIds.has(interest.id)) {
        selectedInterestIds.delete(interest.id);
      } else {
        selectedInterestIds.add(interest.id);
      }

      syncInterestButtonsToState(); 
      applyFiltersToCards(); // immediate show/hide feedback is fine
      scheduleSortUpdate(500); // sort later, only if interest-based

      // cancel pending debounced hash update from note
      if (liveUpdateTimeout) {
        clearTimeout(liveUpdateTimeout);
        liveUpdateTimeout = null;
      }

      writeHashFromUI();
      indicateStateChanged();
    });

    container.appendChild(btn);
  });

  syncInterestButtonsToState();
}

function getKinkMaxRank(kinkId) {
  const activePills = document.querySelectorAll(
    `.role-row[data-kink-id="${kinkId}"] .interest-pill.active`
  );

  if (activePills.length === 0) return 0; // neutral / unknown

  let maxRank = Number.NEGATIVE_INFINITY;

  for (const pill of activePills) {
    const iid = pill.dataset.interestId;
    const rank = interestRankById.get(iid) ?? 0;
    if (rank > maxRank) maxRank = rank;
  }

  return maxRank;
}


function kinkMatchesInterestFilter(kinkId) {
  if (selectedInterestIds.size === 0) return true;

  // pass if ANY role in this kink has an active interest that is selected in the filter
  const activePills = document.querySelectorAll(
    `.role-row[data-kink-id="${kinkId}"] .interest-pill.active`
  );

  for (const pill of activePills) {
    const iid = pill.dataset.interestId;
    if (iid && selectedInterestIds.has(iid)) return true;
  }
  return false;
}

function syncInterestButtonsToState() {
  const container = document.getElementById("interest-filter-buttons");
  if (!container) return;

  container.querySelectorAll("button").forEach((btn) => {
    const id = btn.dataset.interestId || "";

    // All active when none selected
    if (id === "") {
      const allActive = selectedInterestIds.size === 0;
      btn.classList.toggle("active", allActive);
      btn.classList.toggle("btn-primary", allActive);
      btn.classList.toggle("btn-outline-primary", !allActive);
      return;
    }

    const defCls = btn.dataset.defaultClass || "btn-outline-secondary";
    const actCls = btn.dataset.activeClass || "btn-secondary";

    const isActive = selectedInterestIds.has(id);
    btn.classList.toggle("active", isActive);

    if (isActive) {
      btn.classList.add(actCls);
      btn.classList.remove(defCls);
    } else {
      btn.classList.add(defCls);
      btn.classList.remove(actCls);
    }
  });
}

function syncCategoryButtonsToState() {
  const container = document.getElementById("category-filter-buttons");
  if (!container) return;

  container.querySelectorAll("button").forEach((btn) => {
    const btnCat = btn.dataset.categoryId || "";

    // "All" is active when nothing selected
    if (btnCat === "") {
      const allActive = selectedCategoryIds.size === 0;
      btn.classList.toggle("active", allActive);
      btn.classList.toggle("btn-primary", allActive);
      btn.classList.toggle("btn-outline-primary", !allActive);
      return;
    }

    const isActive = selectedCategoryIds.has(btnCat);
    btn.classList.toggle("active", isActive);
    btn.classList.toggle("btn-primary", isActive);
    btn.classList.toggle("btn-outline-primary", !isActive);
  });
}

function sortCards() {
  const row = document.getElementById("kink-row");
  if (!row) return;

  withScrollAnchor(() => {
    const cols = Array.from(row.querySelectorAll(":scope > .col"));
    cols.sort((a, b) => {
      const cardA = a.querySelector(".kink-card");
      const cardB = b.querySelector(".kink-card");

      const nameA = cardA?.dataset.kinkName || "";
      const nameB = cardB?.dataset.kinkName || "";

      if (sortMode === "name_asc") return nameA.localeCompare(nameB);

      const rankA = getKinkMaxRank(cardA?.dataset.kinkId || "");
      const rankB = getKinkMaxRank(cardB?.dataset.kinkId || "");

      if (sortMode === "interest_desc") {
        if (rankB !== rankA) return rankB - rankA;
        return nameA.localeCompare(nameB);
      }
      if (sortMode === "interest_asc") {
        if (rankA !== rankB) return rankA - rankB;
        return nameA.localeCompare(nameB);
      }
      return nameA.localeCompare(nameB);
    });

    for (const col of cols) row.appendChild(col);
  });

  scheduleEqualizeHeaders(50);
}

// ---------- Rendering ----------
function renderKinks() {
  const container = document.getElementById("kink-row");
  if (!container) return;

  container.innerHTML = "";

  kinkData.forEach((kink) => {

    const card = document.createElement("div");
    card.className = "card h-100 kink-card shadow-sm";
    card.dataset.kinkId = kink.id;

    card.dataset.kinkName = (kink.name || "").toLowerCase();

    const catId = kink.category || "";
    card.dataset.categoryId = catId;

    const body = document.createElement("div");
    body.className = "card-body kink-card-body";

    const titleRow = document.createElement("div");
    titleRow.className =
      "d-flex justify-content-between align-items-start gap-2 mb-1";

    const titleLeft = document.createElement("div");

    const titleText = document.createElement("h5");
    titleText.className = "card-title mb-1";
    titleText.textContent = kink.name;

    const subtitle = document.createElement("div");
    subtitle.className = "small";
    subtitle.textContent = kink.id;

    titleLeft.appendChild(titleText);
    if (isDevMode()) {
      titleLeft.appendChild(subtitle);
    }
    const rightBadgeBox = document.createElement("div");
    rightBadgeBox.className = "d-flex flex-column align-items-end gap-1";

    const cat = catId ? categoryById.get(catId) : null;
    const catBadge = document.createElement("span");
    catBadge.className = "badge bg-info-subtle text-info-emphasis";
    catBadge.textContent = cat ? cat.name : "Uncategorized";
    if (cat?.description) catBadge.title = cat.description;

    rightBadgeBox.appendChild(catBadge);

    titleRow.appendChild(titleLeft);
    titleRow.appendChild(rightBadgeBox);

    const desc = document.createElement("p");
    desc.className = "card-text small mb-2  kink-description";
    desc.textContent = kink.description || "";

    const rolesWrapper = document.createElement("div");
    rolesWrapper.className = "mt-2";

    const sortedRoles = [...(kink.roles || [])].sort(alphaByName);

    sortedRoles.forEach((role, roleIndex) => {
      const roleRow = document.createElement("div");
      roleRow.className =
        "border-top pt-2 mt-2 d-flex flex-column gap-1 role-row";
      roleRow.dataset.kinkId = kink.id;
      roleRow.dataset.roleId = role.id;
      roleRow.dataset.roleIndex = String(roleIndex);

      const roleHeader = document.createElement("div");
      roleHeader.className =
        "d-flex justify-content-between align-items-center";

      const roleTitle = document.createElement("div");
      if (isDevMode()) {
        roleTitle.innerHTML =
          `<span class="role-name">${role.name}</span> ` +
          `<span class="text-muted small ms-1">(${role.id})</span>`;
      } else {
        roleTitle.innerHTML = `<span class="role-name">${role.name}</span>`;
      }

      roleHeader.appendChild(roleTitle);

      const roleDesc = document.createElement("p");
      roleDesc.className = "mb-1 small role-description";
      roleDesc.textContent = role.description || "";

      const pillContainer = document.createElement("div");
      pillContainer.className = "d-flex flex-wrap gap-1 mt-1 interest-pills";

      interestScale.forEach((interest) => {
        const pill = document.createElement("button");
        pill.type = "button";

        const defaultClass = interest.btn_class || "btn-outline-secondary";
        const activeClass = interest.btn_active_class || "btn-secondary";

        pill.className = `btn btn-sm rounded-pill interest-pill ${defaultClass}`;
        pill.textContent = interest.label;
        pill.title = interest.description || "";
        pill.dataset.interestId = interest.id;
        pill.dataset.defaultClass = defaultClass;
        pill.dataset.activeClass = activeClass;

        // View-only mode: do not allow recipients to change role interest.
        if (isViewOnlyShareMode()) {
          pill.classList.add("btn-readonly");
          pill.setAttribute("aria-disabled", "true");
        }

        // Toggle-to-clear behavior
        pill.addEventListener("click", () => {
          const isAlreadyActive = pill.classList.contains("active");

          pillContainer.querySelectorAll(".interest-pill").forEach((el) => {
            const defCls = el.dataset.defaultClass || "btn-outline-secondary";
            const actCls = el.dataset.activeClass || "btn-secondary";

            el.classList.remove("active", actCls);
            el.classList.add(defCls);
          });

          if (!isAlreadyActive) {
            pill.classList.add("active", activeClass);
            pill.classList.remove(defaultClass);
          }

          if (liveUpdateTimeout) {
            clearTimeout(liveUpdateTimeout);
            liveUpdateTimeout = null;
          }

          applyFiltersToCards();
          scheduleSortUpdate(500);
          writeHashFromUI();
          indicateStateChanged();
        });

        pillContainer.appendChild(pill);
      });

      // Notes
      const notesGroup = document.createElement("div");
      notesGroup.className = "mt-2";

      const notesId = `note-${kink.id}-${role.id}`;

      const notesLabel = document.createElement("label");
      notesLabel.className = "form-label small mb-1";
      notesLabel.setAttribute("for", notesId);
      notesLabel.textContent = "Notes (optional)";

      const notesInput = document.createElement("textarea");
      notesInput.className = "form-control form-control-sm role-notes";
      notesInput.rows = 2;
      notesInput.id = notesId;
      notesInput.placeholder = "Add details, boundaries, exceptions…";

      // View-only mode: do not allow recipients to edit notes.
      if (isViewOnlyShareMode()) {
        notesInput.readOnly = true;
        notesInput.setAttribute("aria-readonly", "true");
        // Avoid trapping keyboard users in an uneditable field.
        notesInput.tabIndex = -1;
      }

      // In view-only mode, the textarea is read-only, so this will not fire.
      notesInput.addEventListener("input", () => {
        scheduleLiveHashUpdate(400);
      });


      notesGroup.appendChild(notesLabel);
      notesGroup.appendChild(notesInput);

      const roleMeta = document.createElement("div");
      roleMeta.className = "kink-role-meta";

      roleMeta.appendChild(roleHeader);
      roleMeta.appendChild(roleDesc);

    roleRow.appendChild(roleMeta);
    roleRow.appendChild(pillContainer);
    roleRow.appendChild(notesGroup);

      rolesWrapper.appendChild(roleRow);
    });

    const headerWrap = document.createElement("div");
    headerWrap.className = "kink-header";
    headerWrap.appendChild(titleRow);
    headerWrap.appendChild(desc);

    body.appendChild(headerWrap);
    body.appendChild(rolesWrapper);


    card.appendChild(body);

    const col = document.createElement("div");
    col.className = "col";
    col.appendChild(card);
    container.appendChild(col);
  });

  applyFiltersToCards();
  sortCards();
  scheduleEqualizeHeaders(50);
}

// ---------- State <-> UI ----------
function readStateFromUI() {
  const selections = [];

  document.querySelectorAll(".role-row").forEach((row) => {
    const kinkId = row.dataset.kinkId;
    const roleId = row.dataset.roleId;

    const activePill = row.querySelector(".interest-pill.active");
    const interestId = activePill ? activePill.dataset.interestId : "";

    const notesEl = row.querySelector(".role-notes");
    const notes = notesEl ? notesEl.value.trim() : "";

    // Keep your current behavior: store if interest OR notes exists
    if (!interestId && !notes) return;

    selections.push({ kinkId, roleId, interestId, notes });
  });

  return {
    version: 4,
    ui: {
      selectedOnly: selectedOnlyFilterEnabled,
      showDescriptions: showKinkDescriptions,
      sort: sortMode,
      categories: Array.from(selectedCategoryIds).sort(),
      interests: Array.from(selectedInterestIds).sort(),
    },
    selections,
  };
}

function applyStateToUI(state) {
  if (!state || !Array.isArray(state.selections)) return;

  // ---- Restore UI toggle state (backward compatible) ----
  selectedOnlyFilterEnabled = Boolean(state?.ui?.selectedOnly);

  const toggle = document.getElementById("filter-selected-only");
  if (toggle) {
    toggle.checked = selectedOnlyFilterEnabled;
  }

  showKinkDescriptions = state?.ui?.showDescriptions !== false;

  const descToggle = document.getElementById("toggle-descriptions");
  if (descToggle) {
    descToggle.checked = showKinkDescriptions;
  }

  applyDescriptionVisibility();

  // ---- Restore multi-select category filters ----
  selectedCategoryIds = new Set(
    Array.isArray(state?.ui?.categories) ? state.ui.categories : []
  );
  syncCategoryButtonsToState();

  // ---- Restore multi-select interest filters ----
  selectedInterestIds = new Set(
    Array.isArray(state?.ui?.interests) ? state.ui.interests : []
  );
  syncInterestButtonsToState();

  // ---- Restore sort mode ----
  sortMode = typeof state?.ui?.sort === "string" ? state.ui.sort : "name_asc";
  const sortEl = document.getElementById("sort-mode");
  if (sortEl) {
    sortEl.value = sortMode;
  }

  // ---- Reset all pills & notes ----
  document.querySelectorAll(".role-row").forEach((row) => {
    row.querySelectorAll(".interest-pill").forEach((el) => {
      const defCls = el.dataset.defaultClass || "btn-outline-secondary";
      const actCls = el.dataset.activeClass || "btn-secondary";
      el.classList.remove("active", actCls);
      el.classList.add(defCls);
    });

    const notesEl = row.querySelector(".role-notes");
    if (notesEl) notesEl.value = "";
  });

  // ---- Apply saved selections ----
  state.selections.forEach((sel) => {
    const row = document.querySelector(
      `.role-row[data-kink-id="${sel.kinkId}"][data-role-id="${sel.roleId}"]`
    );
    if (!row) return;

    if (sel.interestId) {
      const pill = row.querySelector(
        `.interest-pill[data-interest-id="${sel.interestId}"]`
      );
      if (pill) {
        const defCls = pill.dataset.defaultClass || "btn-outline-secondary";
        const actCls = pill.dataset.activeClass || "btn-secondary";
        pill.classList.add("active", actCls);
        pill.classList.remove(defCls);
      }
    }

    if (typeof sel.notes === "string") {
      const notesEl = row.querySelector(".role-notes");
      if (notesEl) notesEl.value = sel.notes;
    }
  });

  // Apply sorting + filters after state is restored
  sortCards();
  applyFiltersToCards();
}


async function resetFormAndUrl() {
  const activeShare = loadActiveShare();

  if (activeShare) {
    const ok = confirm(
      "Delete this shared link and clear all selections and notes? This cannot be undone."
    );
    if (!ok) return;

    try {
      await deleteShare(activeShare.id, activeShare.manageToken);
      indicateShareStatus("Shared link deleted.");
    } catch (err) {
      console.warn("Share delete failed", err);
      indicateShareStatus("Failed to delete shared data. Please try again.");
      return;
    }
  } else {
    if (!confirm("Clear all selections and notes?")) {
      return;
    }
  }

  // Clear any active share capability in this tab.
  clearActiveShare();

  // Reset URL to the base path (no query params, no fragments).
  history.replaceState(null, "", window.location.pathname);

  document.querySelectorAll(".interest-pill").forEach((pill) => {
    const defCls = pill.dataset.defaultClass || "btn-outline-secondary";
    const actCls = pill.dataset.activeClass || "btn-secondary";

    pill.classList.remove("active", actCls);
    pill.classList.add(defCls);
  });

  document.querySelectorAll(".role-notes").forEach((notesEl) => {
    notesEl.value = "";
  });

  showKinkDescriptions = true;

  const descToggle = document.getElementById("toggle-descriptions");
  if (descToggle) {
    descToggle.checked = true;
  }

  applyDescriptionVisibility();

  if (liveUpdateTimeout) {
    clearTimeout(liveUpdateTimeout);
    liveUpdateTimeout = null;
  }

  updateResetButtonUi();
  indicateShareStatus("Form reset.");
}


function kinkHasSelectedInterest(kinkId) {
  // interest selected means: any .role-row under this kink has an active pill
  return Boolean(
    document.querySelector(
      `.role-row[data-kink-id="${kinkId}"] .interest-pill.active`
    )
  );
}

function applyFiltersToCards() {
  const doWork = () => {
    document.querySelectorAll(".kink-card").forEach((card) => {
      const cardCat = card.dataset.categoryId || "";
      const kinkId = card.dataset.kinkId || "";

      const passesCategory =
        selectedCategoryIds.size === 0 || selectedCategoryIds.has(cardCat);

      const passesSelectedOnly =
        !selectedOnlyFilterEnabled || kinkHasSelectedInterest(kinkId);

      const passesInterest = kinkMatchesInterestFilter(kinkId);
      const passesSearch = kinkMatchesSearch(kinkId);

      const shouldShow =
        passesCategory && passesSelectedOnly && passesInterest && passesSearch;

      const col = card.closest(".col");
      if (col) col.classList.toggle("d-none", !shouldShow);
    });
  };

  // Only anchor-preserve when the sort depends on interest (the jumpy case)
  if (sortMode !== "name_asc") {
    withScrollAnchor(doWork);
  } else {
    doWork();
  }

  scheduleEqualizeHeaders(50);
}


function applyStateFromHash() {
  // If someone opened an older link, automatically rewrite it into the new
  // shorter lz: format (when LZString is available).
  maybeUpgradeLegacyHashToLz();
  const state = decodeStateFromHash(window.location.hash);
  if (state) applyStateToUI(state);
}

async function applyStateFromUrl() {
  const params = new URLSearchParams(window.location.search);
  // Drop any previously-stored manage capability if the user navigates to a
  // different share (or off share pages entirely). Note: after a management URL
  // is verified, we rewrite the address bar to the public share URL for the SAME
  // share id, so we must keep the capability for that share while browsing it.
  const active = loadActiveShare();
  const shareIdInUrl = params.get("s") || params.get("m");
  const isManageUrl = params.has("m");
  const tokenInHash = window.location.hash ? window.location.hash.slice(1) : "";

  if (active) {
    const movedOffShare = !shareIdInUrl;
    const movedToDifferentShare = Boolean(shareIdInUrl) && shareIdInUrl !== active.id;
    const manageTokenMismatch =
      Boolean(shareIdInUrl) &&
      shareIdInUrl === active.id &&
      isManageUrl &&
      (!tokenInHash || tokenInHash !== active.manageToken);

    if (movedOffShare || movedToDifferentShare || manageTokenMismatch) {
      clearActiveShare();
    }
  }
  const shareId = params.get("s") || params.get("m");
  const isManage = params.has("m");

  if (shareId) {
    // Management tokens are kept in the URL fragment so they are not sent to
    // the server in the initial page request.
    const token = window.location.hash ? window.location.hash.slice(1) : "";
    if (isManage && token) {
      saveActiveShare({ id: shareId, manageToken: token });
    }

    const state = await fetchShareState(shareId);
    if (state) {
      applyStateToUI(state);
      // Ensure the address bar is always safe to copy. If the user opened a
      // management link, keep their capability in sessionStorage, but rewrite
      // the visible URL to the public share link.
      history.replaceState(null, "", buildPublicShareUrl(shareId));
      updateResetButtonUi();
      return;
    }

    // If the share API is unavailable, fall back to legacy hash behavior.
    indicateShareStatus("Could not load share link; falling back to legacy URL.");
  }

  applyStateFromHash();
  updateResetButtonUi();
}

// ---------- Share / Copy ----------
let shareStatusTimeout = null;

function indicateShareStatus(message) {
  const statusEl = document.getElementById("share-status");
  if (!statusEl) return;

  statusEl.textContent = message;
  if (shareStatusTimeout) clearTimeout(shareStatusTimeout);
  shareStatusTimeout = setTimeout(() => {
    statusEl.textContent = "";
  }, 3000);
}

let backendSaveTimeout = null;

function scheduleBackendSave(delayMs = 800) {
  const active = loadActiveShare();
  if (!active) return;

  // If the user is truly view-only, never attempt to save.
  // (In practice, activeShare implies manage capability.)
  if (isViewOnlyShareMode()) return;

  if (backendSaveTimeout) {
    clearTimeout(backendSaveTimeout);
  }

  backendSaveTimeout = setTimeout(async () => {
    backendSaveTimeout = null;

    try {
      const state = readStateFromUI();
      await updateShare(active.id, active.manageToken, state);
      indicateShareStatus("Selections saved.");
    } catch (err) {
      console.warn("Auto-save failed", err);
      indicateShareStatus("Failed to save changes. Click Share… to retry.");
    }
  }, delayMs);
}

function indicateStateChanged() {
  const params = new URLSearchParams(window.location.search);
  const inShareMode =
    params.has("s") || params.has("m") || Boolean(loadActiveShare());

  if (inShareMode) {
    // Management mode: actually persist to backend (debounced).
    if (hasManageCapability()) {
      indicateShareStatus("Saving…");
      scheduleBackendSave(800);
    } else {
      // Public share view: they can't save.
    }
  } else {
    indicateShareStatus("Link updated.");
  }
}

function loadActiveShare() {
  const raw = sessionStorage.getItem(SHARE_SESSION_KEY);
  if (!raw) return null;
  try {
    const parsed = JSON.parse(raw);
    if (parsed && typeof parsed.id === "string" && typeof parsed.manageToken === "string") {
      return parsed;
    }
    return null;
  } catch {
    return null;
  }
}

function saveActiveShare(activeShare) {
  sessionStorage.setItem(SHARE_SESSION_KEY, JSON.stringify(activeShare));
}

function clearActiveShare() {
  sessionStorage.removeItem(SHARE_SESSION_KEY);
}

function buildPublicShareUrl(shareId) {
  const url = new URL(window.location.href);
  url.hash = "";
  url.search = "";
  url.searchParams.set("s", shareId);
  return url.toString();
}

function buildManageShareUrl(shareId, manageToken) {
  const url = new URL(window.location.href);
  url.hash = "";
  url.search = "";
  url.searchParams.set("m", shareId);
  url.hash = manageToken;
  return url.toString();
}


function hasManageCapability() {
  return Boolean(loadActiveShare());
}

function isViewOnlyShareMode() {
  // View-only mode means the user loaded a public share link (?s=...) and does
  // NOT have a management capability stored in sessionStorage.
  const params = new URLSearchParams(window.location.search);
  return params.has("s") && !hasManageCapability();
}

function updateResetButtonUi() {
  const btn = document.getElementById("reset-btn");
  if (!btn) return;

  if (hasManageCapability()) {
    btn.textContent = "Delete shared data";
    btn.classList.add("btn-danger");
    btn.classList.remove("btn-outline-secondary");
    btn.setAttribute("aria-label", "Delete shared data");
  } else {
    btn.textContent = "Reset Form";
    btn.classList.add("btn-secondary");
    btn.classList.remove("btn-danger");
    btn.setAttribute("aria-label", "Reset form");
  }
}


async function copyTextToClipboard(text, successMessage) {
  try {
    await navigator.clipboard.writeText(text);
    indicateShareStatus(successMessage);
    return true;
  } catch (err) {
    console.error(err);
    indicateShareStatus("Could not copy (clipboard permissions?).");
    return false;
  }
}

function getShareModal() {
  const el = document.getElementById("share-modal");
  if (!el || !window.bootstrap?.Modal) return null;
  return window.bootstrap.Modal.getOrCreateInstance(el);
}

function updateShareModalLinks(shareId, manageToken) {
  const publicUrl = buildPublicShareUrl(shareId);
  const manageUrl = manageToken ? buildManageShareUrl(shareId, manageToken) : "";

  const publicInput = document.getElementById("share-public-url");
  if (publicInput) publicInput.value = publicUrl;

  const telegramEl = document.getElementById("share-telegram");
  if (telegramEl) {
    const tg = new URL("https://t.me/share/url");
    tg.searchParams.set("url", publicUrl);
    telegramEl.setAttribute("href", tg.toString());
  }

  const manageInput = document.getElementById("share-manage-url");
  const manageDetails = document.getElementById("share-manage-details");
  if (manageInput) manageInput.value = manageUrl;
  if (manageDetails) {
    // Hide management section if we don't have a token (e.g., legacy fallback).
    manageDetails.classList.toggle("d-none", !manageUrl);
  }
}

async function createShare(state) {
  const res = await fetch(SHARE_API_BASE, {
    method: "POST",
    headers: { "Content-Type": "application/json" },
    body: JSON.stringify({ state }),
  });
  if (!res.ok) {
    throw new Error(`Create share failed: ${res.status}`);
  }
  return res.json();
}

async function updateShare(shareId, manageToken, state) {
  const res = await fetch(`${SHARE_API_BASE}/${encodeURIComponent(shareId)}`, {
    method: "PUT",
    headers: {
      "Content-Type": "application/json",
      Authorization: `Bearer ${manageToken}`,
    },
    body: JSON.stringify({ state }),
  });
  if (!res.ok) {
    throw new Error(`Update share failed: ${res.status}`);
  }
  return res.json().catch(() => ({}));
}


async function deleteShare(shareId, manageToken) {
  const res = await fetch(`${SHARE_API_BASE}/${encodeURIComponent(shareId)}`, {
    method: "DELETE",
    headers: {
      Accept: "application/json",
      Authorization: `Bearer ${manageToken}`,
    },
  });
  if (!res.ok) {
    throw new Error(`Delete share failed: ${res.status}`);
  }
  return res.json().catch(() => ({}));
}

async function fetchShareState(shareId) {
  try {
    const res = await fetch(`${SHARE_API_BASE}/${encodeURIComponent(shareId)}`, {
      method: "GET",
      headers: { Accept: "application/json" },
    });
    if (!res.ok) return null;
    const data = await res.json();
    if (!data || typeof data !== "object") return null;
    if (!data.state) return null;
    return data.state;
  } catch (err) {
    console.warn("Share fetch failed", err);
    return null;
  }
}

async function openShareModalForCurrentState() {
  const modal = getShareModal();
  if (!modal) {
    indicateShareStatus("Share modal unavailable.");
    return;
  }

  // Always start from a clean modal state.
  const viewOnlyActions = document.getElementById("share-viewonly-actions");
  if (viewOnlyActions) viewOnlyActions.classList.add("d-none");

  const state = readStateFromUI();
  const existing = loadActiveShare();

  const params = new URLSearchParams(window.location.search);
  const shareIdFromUrl = params.get("s");
  const isViewingPublicShare = typeof shareIdFromUrl === "string" && shareIdFromUrl.length > 0;

  // If the user is viewing a public share link (someone else's link) and does
  // not have a management capability, do not create a new share automatically.
  // Instead, show the existing public link and offer an explicit "create my own" action.
  if (isViewingPublicShare && !existing) {
    updateShareModalLinks(shareIdFromUrl, null);

    const manageDetails = document.getElementById("share-manage-details");
    if (manageDetails) manageDetails.classList.add("d-none");

    if (viewOnlyActions) viewOnlyActions.classList.remove("d-none");

    const createOwnBtn = document.getElementById("share-create-own");
    if (createOwnBtn) {
      createOwnBtn.onclick = async () => {
        try {
          createOwnBtn.disabled = true;
          createOwnBtn.setAttribute("aria-busy", "true");

          const latestState = readStateFromUI();
          const created = await createShare(latestState);
          const newShareId = created?.id;
          const manageToken = created?.manageToken;

          if (typeof newShareId !== "string" || typeof manageToken !== "string") {
            throw new Error("Invalid create-share response.");
          }

          saveActiveShare({ id: newShareId, manageToken });
          updateResetButtonUi();
          updateShareModalLinks(newShareId, manageToken);
          history.replaceState(null, "", buildPublicShareUrl(newShareId));

          if (viewOnlyActions) viewOnlyActions.classList.add("d-none");
          const md = document.getElementById("share-manage-details");
          if (md) md.classList.remove("d-none");

          indicateShareStatus("Share link created.");
        } catch (err) {
          console.warn(err);
          indicateShareStatus("Failed to create share link. Please try again.");
        } finally {
          createOwnBtn.disabled = false;
          createOwnBtn.removeAttribute("aria-busy");
        }
      };
    }

    modal.show();
    return;
  }

  // Preferred path: update existing share when we have a management capability.
  if (existing) {
    try {
      await updateShare(existing.id, existing.manageToken, state);
      updateShareModalLinks(existing.id, existing.manageToken);
      history.replaceState(null, "", buildPublicShareUrl(existing.id));
      modal.show();
      return;
    } catch (err) {
      console.warn(err);
      indicateShareStatus("Could not update share; creating a new one.");
      clearActiveShare();
    }
  }

  // Create a new share.
  try {
    const created = await createShare(state);
    const shareId = created?.id;
    const manageToken = created?.manageToken;

    if (typeof shareId !== "string" || typeof manageToken !== "string") {
      throw new Error("Invalid create-share response.");
    }

    saveActiveShare({ id: shareId, manageToken });
    updateResetButtonUi();
    updateShareModalLinks(shareId, manageToken);
    history.replaceState(null, "", buildPublicShareUrl(shareId));

    const manageDetails = document.getElementById("share-manage-details");
    if (manageDetails) manageDetails.classList.remove("d-none");

    modal.show();
  } catch (err) {
    // Graceful fallback: show a legacy share URL (hash-based) when the API
    // isn't available yet.
    console.warn(err);
    const legacyHash = encodeStateToHash(state);
    const legacyUrl = new URL(window.location.href);
    legacyUrl.search = "";
    legacyUrl.hash = legacyHash.startsWith("#") ? legacyHash.slice(1) : legacyHash;

    const publicInput = document.getElementById("share-public-url");
    if (publicInput) publicInput.value = legacyUrl.toString();

    const telegramEl = document.getElementById("share-telegram");
    if (telegramEl) {
      const tg = new URL("https://t.me/share/url");
      tg.searchParams.set("url", legacyUrl.toString());
      telegramEl.setAttribute("href", tg.toString());
    }

    const manageDetails = document.getElementById("share-manage-details");
    if (manageDetails) manageDetails.classList.add("d-none");

    modal.show();
    indicateShareStatus("Using legacy share URL (share API not configured). ");
  }
}

// ---------- Boot ----------

document.addEventListener("DOMContentLoaded", () => {
  setDobMaxToday();

  // If the user loaded a legacy share link, upgrade it immediately so that
  // subsequent copying/sharing produces the shorter format.
  maybeUpgradeLegacyHashToLz();

  document
    .getElementById("age-gate-form")
    ?.addEventListener("submit", handleAgeGateSubmit);

  // Dev-only reset age gate button
  const resetBtn = document.getElementById("reset-age-btn");
  if (resetBtn) {
    if (isDevMode()) resetBtn.classList.remove("d-none");
    resetBtn.addEventListener("click", resetAgeCheck);
  }

  const ageVerified = sessionStorage.getItem(AGE_SESSION_KEY) === "true";
  if (ageVerified) {
    showApp();
    void initApp();
  } else {
    showAgeGate();
  }

  document
    .getElementById("filter-selected-only")
    ?.addEventListener("change", (e) => {
      selectedOnlyFilterEnabled = Boolean(e.target.checked);
      applyFiltersToCards();
      scheduleSortUpdate(500);

      // Keep the URL in sync
      if (liveUpdateTimeout) {
        clearTimeout(liveUpdateTimeout);
        liveUpdateTimeout = null;
      }
      writeHashFromUI();
      indicateStateChanged();
    });
  document
  .getElementById("toggle-descriptions")
  ?.addEventListener("change", (e) => {
    showKinkDescriptions = Boolean(e.target.checked);
    applyDescriptionVisibility();

    if (liveUpdateTimeout) {
      clearTimeout(liveUpdateTimeout);
      liveUpdateTimeout = null;
    }

    writeHashFromUI();
    indicateStateChanged();
  });

  document
    .getElementById("share-btn")
    ?.addEventListener("click", () => {
      void openShareModalForCurrentState();
    });

  document.getElementById("share-public-copy")?.addEventListener("click", () => {
    const input = document.getElementById("share-public-url");
    const text = input ? input.value : "";
    if (text) {
      void copyTextToClipboard(text, "Share link copied.");
    }
  });

  document.getElementById("share-manage-copy")?.addEventListener("click", () => {
    const input = document.getElementById("share-manage-url");
    const text = input ? input.value : "";
    if (text) {
      void copyTextToClipboard(text, "Management link copied.");
    }
  });
  document
    .getElementById("reset-btn")
    ?.addEventListener("click", resetFormAndUrl);
  document.getElementById("sort-mode")?.addEventListener("change", (e) => {
    sortMode = e.target.value || "name_asc";
    // Cancel any pending scheduled sort
    if (sortUpdateTimeout) {
      clearTimeout(sortUpdateTimeout);
      sortUpdateTimeout = null;
    }

    // Apply immediately when the user explicitly changes sort mode
    sortCards();
    applyFiltersToCards();

    // Keep URL in sync
    if (liveUpdateTimeout) {
      clearTimeout(liveUpdateTimeout);
      liveUpdateTimeout = null;
    }
    writeHashFromUI();
    indicateStateChanged();
  });

  window.addEventListener("hashchange", () => {
    if (!document.getElementById("app-root")?.classList.contains("d-none")) {
      const params = new URLSearchParams(window.location.search);
      if (params.has("s") || params.has("m")) {
        // In share-link mode, the fragment (if any) is a management token and
        // should not be interpreted as legacy state.
        return;
      }
      maybeUpgradeLegacyHashToLz();
      applyStateFromHash();
    }
  });
    window.addEventListener("resize", () => {
    scheduleEqualizeHeaders(120);
  });

});
