#!/usr/bin/env python3
from __future__ import annotations

import argparse
import difflib
import string
import sys
import re
from dataclasses import dataclass
from pathlib import Path
from typing import Any

from ruamel.yaml import YAML
from ruamel.yaml.comments import CommentedMap, CommentedSeq

try:
    from jsonschema import Draft202012Validator
    from jsonschema.exceptions import ValidationError as JsonSchemaValidationError
except Exception as exc:  # noqa: BLE001
    raise SystemExit(
        "ERROR: Missing dependency 'jsonschema'. Install with: pip install jsonschema"
    ) from exc


WWWROOT_DIRNAME = "wwwroot"
TOOLS_DIRNAME = "tools"


def find_project_root(start: Path) -> Path | None:
    cwd = start.resolve()

    if (cwd / WWWROOT_DIRNAME).is_dir():
        return cwd

    if cwd.name == TOOLS_DIRNAME and (cwd.parent / WWWROOT_DIRNAME).is_dir():
        return cwd.parent

    return None


class ValidationError(Exception):
    """User-facing validation error with location hints."""


@dataclass(frozen=True)
class YamlFile:
    path: Path
    root_key: str


# ----------------------------
# JSON Schemas (always enforced)
# ----------------------------

INTEREST_SCALE_SCHEMA: dict[str, Any] = {
    "$schema": "https://json-schema.org/draft/2020-12/schema",
    "type": "object",
    "additionalProperties": False,
    "required": ["interest_scale"],
    "properties": {
        "interest_scale": {
            "type": "array",
            "items": {
                "type": "object",
                "additionalProperties": False,
                "required": [
                    "id",
                    "label",
                    "rank",
                    "description",
                    "btn_class",
                    "btn_active_class",
                ],
                "properties": {
                    "id": {"type": "string", "minLength": 1},
                    "label": {"type": "string", "minLength": 1},
                    "rank": {"type": "integer"},
                    "description": {"type": "string", "minLength": 1},
                    "btn_class": {"type": "string", "minLength": 1},
                    "btn_active_class": {"type": "string", "minLength": 1},
                },
            },
        }
    },
}

KINK_CATEGORIES_SCHEMA: dict[str, Any] = {
    "$schema": "https://json-schema.org/draft/2020-12/schema",
    "type": "object",
    "additionalProperties": False,
    "required": ["categories"],
    "properties": {
        "categories": {
            "type": "array",
            "items": {
                "type": "object",
                "additionalProperties": False,
                "required": ["id", "name", "description"],
                "properties": {
                    "id": {"type": "string", "minLength": 1},
                    "name": {"type": "string", "minLength": 1},
                    "description": {"type": "string", "minLength": 1},
                },
            },
        }
    },
}

KINKS_SCHEMA: dict[str, Any] = {
    "$schema": "https://json-schema.org/draft/2020-12/schema",
    "type": "object",
    "additionalProperties": False,
    "required": ["kinks"],
    "properties": {
        "kinks": {
            "type": "array",
            "items": {
                "type": "object",
                "additionalProperties": False,
                "required": ["id", "name", "category", "description", "roles"],
                "properties": {
                    "id": {"type": "string", "minLength": 1},
                    "name": {"type": "string", "minLength": 1},
                    "category": {"type": "string", "minLength": 1},
                    "description": {"type": "string", "minLength": 1},
                    "roles": {
                        "type": "array",
                        "minItems": 1,
                        "items": {
                            "type": "object",
                            "additionalProperties": False,
                            "required": ["id", "name", "description"],
                            "properties": {
                                "id": {"type": "string", "minLength": 1},
                                "name": {"type": "string", "minLength": 1},
                                "description": {"type": "string", "minLength": 1},
                            },
                        },
                    },
                },
            },
        }
    },
}


# ----------------------------
# Utilities
# ----------------------------

_ID_SEP_RE = re.compile(r"[\s\\-]+")
_MULTI_UNDERSCORE_RE = re.compile(r"_+")
TRAILING_NAME_PUNCT_RE = re.compile(r"[.,:;!?]+$")
# Collapse all whitespace (including newlines/tabs) to single spaces.
_DEDUPE_WS_RE = re.compile(r"\s+")


def _dedupe_whitespace(value: str) -> str:
    """Collapse runs of whitespace into single spaces and strip."""
    return _DEDUPE_WS_RE.sub(" ", value).strip()


def _strip_trailing_punctuation(value: str) -> str:
    """Remove punctuation characters from the end of a string, but keep a trailing ')' and/or `"`."""
    v = _dedupe_whitespace(value)
    if not v:
        return v

    # Handle both ASCII punctuation and the common ellipsis character.
    # Keep ending ')' for acronyms.
    punct = (set(string.punctuation) - {")", '"'}) | {"…"}

    while v and v[-1] in punct:
        v = v[:-1].rstrip()

    if v.endswith(")") and "(" not in v:
        v = v.rstrip(")").rstrip()
    if v.endswith('"') and v.count('"') % 2 == 1:
        v = v.rstrip('"').rstrip()

    return v


def _yaml() -> YAML:
    y = YAML(typ="rt")  # round-trip
    y.preserve_quotes = True
    y.width = 4096
    y.indent(mapping=2, sequence=4, offset=2)
    return y


def _die(msg: str) -> None:
    print(f"ERROR: {msg}", file=sys.stderr)
    raise SystemExit(1)


def _warn(msg: str) -> None:
    print(f"WARNING: {msg}", file=sys.stderr)


def _require_file(path: Path) -> None:
    if not path.exists():
        _die(f"Missing required file: {path}")


def normalize_id(value: str) -> str:
    """
    Lowercase, replace spaces and hyphens with '_', collapse repeats.
    """
    v = value.strip().lower()
    v = _ID_SEP_RE.sub("_", v)
    v = _MULTI_UNDERSCORE_RE.sub("_", v)
    return v.strip("_")


def normalize_title(value: str) -> str:
    """
    Dedupe whitespace, strip any trailing punctuation, and ensure it starts with
    a capital letter.
    """
    v = _strip_trailing_punctuation(value)
    if not v:
        return v
    return (v[0].upper() + v[1:]).strip()


def normalize_description(value: str) -> str:
    """
    Dedupe whitespace, ensure it starts with a capital letter, and always end
    with a period.
    """
    v = _dedupe_whitespace(value)
    if not v:
        return v

    # Normalize the start.
    v = (v[0].upper() + v[1:]).strip()

    # Force a single trailing period.
    v = _strip_trailing_punctuation(v)
    if v:
        v = v.rstrip() + "."
    return v


def _load_yaml(path: Path) -> CommentedMap:
    y = _yaml()
    try:
        data = y.load(path.read_text(encoding="utf-8"))
    except Exception as exc:  # noqa: BLE001
        raise ValidationError(f"Failed to parse YAML: {path} ({exc})") from exc

    if not isinstance(data, CommentedMap):
        raise ValidationError(f"Top-level YAML must be a mapping in {path}")
    return data


def _dump_yaml(data: CommentedMap) -> str:
    y = _yaml()
    import io

    buf = io.StringIO()
    y.dump(data, buf)
    text = buf.getvalue()
    if not text.endswith("\n"):
        text += "\n"
    return text


def _insert_blank_lines_between_seq_items(yaml_text: str, *, list_indent: int) -> str:
    indent_prefix = " " * list_indent + "- "
    lines = yaml_text.splitlines()
    out: list[str] = []

    seen_first_item_at_indent = False
    for line in lines:
        is_item_line = line.startswith(indent_prefix)
        if is_item_line and seen_first_item_at_indent:
            if out and out[-1].strip() != "":
                out.append("")
        if is_item_line:
            seen_first_item_at_indent = True
        out.append(line)

    return "\n".join(out) + "\n"


def _to_plain(obj: Any) -> Any:
    """Convert ruamel round-trip structures into plain dict/list for jsonschema."""
    if isinstance(obj, CommentedMap):
        return {k: _to_plain(v) for k, v in obj.items()}
    if isinstance(obj, CommentedSeq):
        return [_to_plain(v) for v in obj]
    if isinstance(obj, dict):
        return {k: _to_plain(v) for k, v in obj.items()}
    if isinstance(obj, list):
        return [_to_plain(v) for v in obj]
    return obj


def _validate_schema(*, path: Path, data: CommentedMap, schema: dict[str, Any]) -> None:
    validator = Draft202012Validator(schema)
    instance = _to_plain(data)

    errors = sorted(validator.iter_errors(instance), key=lambda e: list(e.path))
    if not errors:
        return

    err = errors[0]
    path_parts = list(err.path)

    # Build readable location
    loc = ".".join(str(p) for p in path_parts) if path_parts else "<root>"

    # Walk up the path to find the nearest mapping (bad object)
    bad_obj = data
    bad_obj_path = []

    try:
        for p in path_parts:
            bad_obj = bad_obj[p]
            bad_obj_path.append(p)
            if isinstance(bad_obj, (dict, CommentedMap)):
                last_mapping = bad_obj
    except Exception:
        last_mapping = None

    snippet = ""
    if isinstance(locals().get("last_mapping"), (dict, CommentedMap)):
        try:
            snippet = _yaml_snippet(last_mapping)  # pyright: ignore[reportPossiblyUnboundVariable]
        except Exception:
            snippet = ""

    message = f"{path}: schema validation failed at {loc}\n{err.message}"

    if snippet:
        message += "\n\nOffending object:\n" + snippet

    raise ValidationError(message)


def _expect_top_list(data: CommentedMap, path: Path, key: str) -> CommentedSeq:
    if key not in data:
        raise ValidationError(f"{path}: missing top-level key '{key}'")
    seq = data[key]
    if not isinstance(seq, CommentedSeq):
        raise ValidationError(f"{path}: '{key}' must be a list")
    return seq


def _expect_str(obj: Any, *, path: Path, ctx: str, field: str) -> str:
    if not isinstance(obj, str) or not obj.strip():
        raise ValidationError(f"{path}: {ctx}: '{field}' must be a non-empty string")
    return obj.strip()


def _expect_int(obj: Any, *, path: Path, ctx: str, field: str) -> int:
    if isinstance(obj, bool):
        raise ValidationError(f"{path}: {ctx}: '{field}' must be an integer, not bool")
    if not isinstance(obj, int):
        raise ValidationError(f"{path}: {ctx}: '{field}' must be an integer")
    return obj


def _check_unique(
    seen: dict[str, dict[Any, str]],
    *,
    path: Path,
    ctx: str,
    field: str,
    value: Any,
) -> None:
    if field not in seen:
        seen[field] = {}
    if value in seen[field]:
        first_ctx = seen[field][value]
        raise ValidationError(
            f"{path}: duplicate {field}={value!r}. First seen at {first_ctx}; duplicate at {ctx}."
        )
    seen[field][value] = ctx

# ----------------------------
# Duplicate description checks
# ----------------------------

# These are commonly reused boilerplate role descriptions that are usually intentional.
# Keeping them in an allowlist reduces noise while still catching more problematic
# copy/paste mistakes.
DEFAULT_ROLE_DESCRIPTION_ALLOWLIST: set[str] = {
    "a general interest.",
    "general interest.",
    "your partner.",
    "yourself.",
}

# Kink-level descriptions are usually expected to be unique; keep this empty unless you
# intentionally want to allow certain repeated descriptions.
DEFAULT_KINK_DESCRIPTION_ALLOWLIST: set[str] = set()


# Role description duplicates are ignored when *all* occurrences are in one of
# these categories (intentional templating).
DUP_ROLE_IGNORE_CATEGORY_IDS: set[str] = {
    "clothing_material",
}


_WHITESPACE_RE = re.compile(r"\s+")


def _dupe_key(value: str) -> str:
    """Normalize a string for duplicate detection (case/whitespace-insensitive)."""
    return _WHITESPACE_RE.sub(" ", value.strip()).casefold()


def _format_dupe_report(
    *,
    kind: str,
    dupes: dict[str, list[str]],
    max_examples: int = 12,
) -> str:
    lines: list[str] = []
    lines.append(f"Duplicate {kind} descriptions detected:")
    for key in sorted(dupes.keys()):
        refs = dupes[key]
        lines.append(f"  - {refs[0].split(':', 1)[0]}: {len(refs)} occurrences")
        for ex in refs[:max_examples]:
            lines.append(f"      • {ex}")
        if len(refs) > max_examples:
            lines.append(f"      • ... and {len(refs) - max_examples} more")
    return "\n".join(lines)


def _check_duplicate_descriptions_in_kinks(
    items: CommentedSeq,
    *,
    path: Path,
    mode: str,
    role_allowlist: set[str] | None = None,
    kink_allowlist: set[str] | None = None,
) -> None:
    """
    Detect duplicate descriptions that are a common sign of copy/paste errors.

    mode:
      - "ignore": do nothing
      - "warn": emit warnings
      - "error": raise ValidationError
    """
    mode = mode.lower().strip()
    if mode not in {"ignore", "warn", "error"}:
        raise ValidationError(
            f"{path}: invalid duplicate description mode {mode!r} "
            "(expected: ignore|warn|error)"
        )

    if mode == "ignore":
        return

    role_allow = { _dupe_key(s) for s in (role_allowlist or DEFAULT_ROLE_DESCRIPTION_ALLOWLIST) }
    kink_allow = { _dupe_key(s) for s in (kink_allowlist or DEFAULT_KINK_DESCRIPTION_ALLOWLIST) }

    kink_desc_map: dict[str, list[str]] = {}
    role_desc_map: dict[str, list[dict[str, str]]] = {}

    for idx, kink in enumerate(items):
        ctx = f"kinks[{idx}]"
        if not isinstance(kink, (dict, CommentedMap)):
            continue

        kid = str(kink.get("id", "")).strip()
        kname = str(kink.get("name", "")).strip()
        kcat = str(kink.get("category", "")).strip()
        kdesc = str(kink.get("description", "")).strip()

        kkey = _dupe_key(kdesc)
        if kkey and kkey not in kink_allow:
            kink_desc_map.setdefault(kkey, []).append(
                f"{ctx}: kink id={kid!r} name={kname!r} description={kdesc!r}"
            )

        roles_any = kink.get("roles", [])
        if not isinstance(roles_any, (list, CommentedSeq)):
            continue

        for r_idx, role in enumerate(roles_any):
            rctx = f"{ctx}.roles[{r_idx}]"
            if not isinstance(role, (dict, CommentedMap)):
                continue

            rid = str(role.get("id", "")).strip()
            rname = str(role.get("name", "")).strip()
            rdesc = str(role.get("description", "")).strip()

            rkey = _dupe_key(rdesc)
            if rkey and rkey not in role_allow:
                role_desc_map.setdefault(rkey, []).append(
                    {
                        "line": f"{rctx}: kink id={kid!r} role id={rid!r} name={rname!r} description={rdesc!r}",
                        "category": kcat,
                    }
                )

    kink_dupes = {k: v for k, v in kink_desc_map.items() if len(v) > 1}
    role_dupes_raw = {k: v for k, v in role_desc_map.items() if len(v) > 1}
    role_dupes: dict[str, list[str]] = {}
    for k, occ in role_dupes_raw.items():
        cats = {o.get("category", "") for o in occ}
        if cats and cats.issubset(DUP_ROLE_IGNORE_CATEGORY_IDS):
            continue
        role_dupes[k] = [o.get("line", "") for o in occ]

    if not kink_dupes and not role_dupes:
        return

    msg_parts: list[str] = [f"{path}:"]

    if kink_dupes:
        msg_parts.append(_format_dupe_report(kind="kink", dupes=kink_dupes))

    if role_dupes:
        msg_parts.append(_format_dupe_report(kind="role", dupes=role_dupes))

    msg = "\n\n".join(msg_parts)

    if mode == "warn":
        _warn(msg)
        return

    raise ValidationError(msg)


# ----------------------------
# Normalizers
# ----------------------------


def normalize_interest_scale(file: YamlFile) -> str:
    data = _load_yaml(file.path)
    _validate_schema(path=file.path, data=data, schema=INTEREST_SCALE_SCHEMA)

    items = _expect_top_list(data, file.path, file.root_key)

    seen: dict[str, dict[Any, str]] = {}
    normalized = CommentedSeq()

    for idx, item in enumerate(items):
        ctx = f"{file.root_key}[{idx}]"
        if not isinstance(item, (dict, CommentedMap)):
            raise ValidationError(f"{file.path}: {ctx}: item must be a mapping")

        iid = normalize_id(
            _expect_str(item.get("id"), path=file.path, ctx=ctx, field="id")
        )
        label = normalize_title(
            _expect_str(item.get("label"), path=file.path, ctx=ctx, field="label")
        )
        desc = normalize_description(
            _expect_str(
                item.get("description"), path=file.path, ctx=ctx, field="description"
            )
        )

        rank = _expect_int(item.get("rank"), path=file.path, ctx=ctx, field="rank")
        btn_class = _expect_str(
            item.get("btn_class"), path=file.path, ctx=ctx, field="btn_class"
        )
        btn_active_class = _expect_str(
            item.get("btn_active_class"),
            path=file.path,
            ctx=ctx,
            field="btn_active_class",
        )

        _check_unique(seen, path=file.path, ctx=ctx, field="id", value=iid)
        _check_unique(seen, path=file.path, ctx=ctx, field="label", value=label)
        _check_unique(seen, path=file.path, ctx=ctx, field="rank", value=rank)
        _check_unique(seen, path=file.path, ctx=ctx, field="description", value=desc)
        _check_unique(seen, path=file.path, ctx=ctx, field="btn_class", value=btn_class)
        _check_unique(
            seen,
            path=file.path,
            ctx=ctx,
            field="btn_active_class",
            value=btn_active_class,
        )

        nm = CommentedMap()
        nm["id"] = iid
        nm["label"] = label
        nm["rank"] = rank
        nm["description"] = desc
        nm["btn_class"] = btn_class
        nm["btn_active_class"] = btn_active_class
        normalized.append(nm)

    normalized.sort(key=lambda x: int(x["rank"]))  # type: ignore[index]

    out = CommentedMap()
    out[file.root_key] = normalized

    text = _dump_yaml(out)
    text = _insert_blank_lines_between_seq_items(text, list_indent=2)
    return text


def normalize_categories(file: YamlFile) -> tuple[str, list[str]]:
    data = _load_yaml(file.path)
    _validate_schema(path=file.path, data=data, schema=KINK_CATEGORIES_SCHEMA)

    items = _expect_top_list(data, file.path, file.root_key)

    seen: dict[str, dict[Any, str]] = {}
    normalized = CommentedSeq()
    category_ids: list[str] = []

    for idx, item in enumerate(items):
        ctx = f"{file.root_key}[{idx}]"
        if not isinstance(item, (dict, CommentedMap)):
            raise ValidationError(f"{file.path}: {ctx}: item must be a mapping")

        cid = normalize_id(
            _expect_str(item.get("id"), path=file.path, ctx=ctx, field="id")
        )
        name = normalize_title(
            _expect_str(item.get("name"), path=file.path, ctx=ctx, field="name")
        )
        desc = normalize_description(
            _expect_str(
                item.get("description"), path=file.path, ctx=ctx, field="description"
            )
        )

        _check_unique(seen, path=file.path, ctx=ctx, field="id", value=cid)
        _check_unique(seen, path=file.path, ctx=ctx, field="name", value=name)
        _check_unique(seen, path=file.path, ctx=ctx, field="description", value=desc)

        nm = CommentedMap()
        nm["id"] = cid
        nm["name"] = name
        nm["description"] = desc
        normalized.append(nm)
        category_ids.append(cid)

    normalized.sort(key=lambda x: str(x["name"]).casefold())  # type: ignore[index]

    out = CommentedMap()
    out[file.root_key] = normalized

    text = _dump_yaml(out)
    text = _insert_blank_lines_between_seq_items(text, list_indent=2)
    return text, category_ids


def normalize_kinks(
    file: YamlFile,
    *,
    valid_category_ids: set[str],
    duplicate_description_mode: str = "warn",
) -> str:
    data = _load_yaml(file.path)
    _validate_schema(path=file.path, data=data, schema=KINKS_SCHEMA)

    items = _expect_top_list(data, file.path, file.root_key)

    seen_global: dict[str, dict[Any, str]] = {}
    normalized = CommentedSeq()

    for idx, item in enumerate(items):
        ctx = f"{file.root_key}[{idx}]"
        if not isinstance(item, (dict, CommentedMap)):
            raise ValidationError(f"{file.path}: {ctx}: item must be a mapping")

        kid = normalize_id(
            _expect_str(item.get("id"), path=file.path, ctx=ctx, field="id")
        )
        name = normalize_title(
            _expect_str(item.get("name"), path=file.path, ctx=ctx, field="name")
        )
        category = normalize_id(
            _expect_str(item.get("category"), path=file.path, ctx=ctx, field="category")
        )
        desc = normalize_description(
            _expect_str(
                item.get("description"), path=file.path, ctx=ctx, field="description"
            )
        )
        if category not in valid_category_ids:
            raise ValidationError(
                f"{file.path}: {ctx} (kink id={kid!r}): category {category!r} "
                f"does not match any category id in kink_categories.yml"
            )

        _check_unique(seen_global, path=file.path, ctx=ctx, field="id", value=kid)
        _check_unique(seen_global, path=file.path, ctx=ctx, field="name", value=name)

        roles_any = item.get("roles")
        if not isinstance(roles_any, (list, CommentedSeq)) or len(roles_any) < 1:
            raise ValidationError(
                f"{file.path}: {ctx} (kink id={kid!r}): 'roles' must be a non-empty list"
            )

        role_seen_ids: set[str] = set()
        role_seen_names: set[str] = set()
        roles_norm = CommentedSeq()

        for r_idx, role in enumerate(roles_any):
            rctx = f"{ctx}.roles[{r_idx}]"
            if not isinstance(role, (dict, CommentedMap)):
                raise ValidationError(f"{file.path}: {rctx}: role must be a mapping")

            rid = normalize_id(
                _expect_str(role.get("id"), path=file.path, ctx=rctx, field="id")
            )
            rname = normalize_title(
                _expect_str(role.get("name"), path=file.path, ctx=rctx, field="name")
            )
            rdesc = normalize_description(
                _expect_str(
                    role.get("description"),
                    path=file.path,
                    ctx=rctx,
                    field="description",
                )
            )

            if rid in role_seen_ids:
                raise ValidationError(
                    f"{file.path}: {ctx} (kink id={kid!r}): duplicate role id {rid!r}"
                )
            if rname in role_seen_names:
                raise ValidationError(
                    f"{file.path}: {ctx} (kink id={kid!r}): duplicate role name {rname!r}"
                )

            role_seen_ids.add(rid)
            role_seen_names.add(rname)

            rm = CommentedMap()
            rm["id"] = rid
            rm["name"] = rname
            rm["description"] = rdesc
            roles_norm.append(rm)

        roles_norm.sort(key=lambda x: str(x["name"]).casefold())  # type: ignore[index]

        km = CommentedMap()
        km["id"] = kid
        km["name"] = name
        km["category"] = category
        km["description"] = desc
        km["roles"] = roles_norm
        normalized.append(km)

    normalized.sort(key=lambda x: str(x["name"]).casefold())  # type: ignore[index]

    # Detect likely copy/paste mistakes (duplicate descriptions). This runs on the
    # normalized data so capitalization/whitespace differences don't hide duplicates.
    _check_duplicate_descriptions_in_kinks(
        normalized,
        path=file.path,
        mode=duplicate_description_mode,
    )


    out = CommentedMap()
    out[file.root_key] = normalized

    text = _dump_yaml(out)
    text = _insert_blank_lines_between_seq_items(text, list_indent=2)
    return text


# ----------------------------
# Prettifiers
# ----------------------------


def prettify_css_text(raw: str) -> str:
    try:
        import cssbeautifier  # type: ignore[import-not-found]
    except Exception:  # noqa: BLE001
        return "\n".join(line.rstrip() for line in raw.splitlines()).strip() + "\n"

    opts = cssbeautifier.default_options()
    opts.indent_size = 2
    opts.end_with_newline = True
    opts.preserve_newlines = True
    opts.max_preserve_newlines = 2

    return cssbeautifier.beautify(raw, opts)


# ----------------------------
# IO helpers
# ----------------------------


def _unified_diff(path: Path, before: str, after: str) -> str:
    return "".join(
        difflib.unified_diff(
            before.splitlines(keepends=True),
            after.splitlines(keepends=True),
            fromfile=str(path),
            tofile=str(path),
        )
    )


def _write_or_check(
    path: Path,
    *,
    new_text: str,
    check: bool,
    show_diff: bool,
) -> bool:
    old_text = path.read_text(encoding="utf-8") if path.exists() else ""
    if old_text == new_text:
        return False

    if show_diff:
        diff = _unified_diff(path, old_text, new_text)
        if diff:
            sys.stdout.write(diff)

    if not check:
        path.write_text(new_text, encoding="utf-8")

    return True


def _yaml_snippet(obj: Any) -> str:
    """
    Render a small YAML snippet for an object without mutating global YAML settings.
    """
    y = YAML()
    y.indent(mapping=2, sequence=4, offset=2)
    y.width = 4096

    import io

    buf = io.StringIO()
    y.dump(obj, buf)
    return buf.getvalue().rstrip()


# ----------------------------
# CLI
# ----------------------------


def main(argv: list[str] | None = None) -> int:
    parser = argparse.ArgumentParser(
        prog="wwwroot_qa",
        description="Validate/sort YAML and prettify HTML/CSS for a wwwroot directory.",
    )

    # Make it optional (0 or 1 args) instead of required.
    parser.add_argument(
        "wwwroot",
        type=Path,
        nargs="?",
        default=None,
        help="Path to wwwroot (optional). If omitted, uses <project_root>/wwwroot.",
    )
    parser.add_argument(
        "--check",
        action="store_true",
        help="Do not write files. Exit with code 2 if changes would be made.",
    )
    parser.add_argument(
        "--diff",
        action="store_true",
        help="Print unified diffs for any file that would change.",
    )

    parser.add_argument(
        "--duplicate-descriptions",
        choices=["ignore", "warn", "error"],
        default="warn",
        help=(
            "How to handle duplicate kink/role descriptions in kinks.yml. "
            "Useful to catch copy/paste mistakes. Default: warn."
        ),
    )

    args = parser.parse_args(argv)

    if args.wwwroot is not None:
        wwwroot = args.wwwroot.resolve()
    else:
        project_root = find_project_root(Path.cwd())
        if project_root is None:
            print(
                f"Refusing to run: could not find a project root containing '{WWWROOT_DIRNAME}/'.\n"
                f"Run from the project root or from '{TOOLS_DIRNAME}/' under it, "
                f"or pass an explicit path to wwwroot.",
                file=sys.stderr,
            )
            return 2
        wwwroot = (project_root / WWWROOT_DIRNAME).resolve()

    if not wwwroot.is_dir():
        print(
            f"Refusing to run: wwwroot path does not exist: {wwwroot}", file=sys.stderr
        )
        return 2

    yml_dir = wwwroot / "yaml"
    if not yml_dir.exists() or not yml_dir.is_dir():
        _die(f"Missing required yml directory: {yml_dir}")

    interest_file = YamlFile(yml_dir / "interest_scale.yml", "interest_scale")
    categories_file = YamlFile(yml_dir / "kink_categories.yml", "categories")
    kinks_file = YamlFile(yml_dir / "kinks.yml", "kinks")

    _require_file(interest_file.path)
    _require_file(categories_file.path)
    _require_file(kinks_file.path)

    changed_any = False

    try:
        interest_text = normalize_interest_scale(interest_file)
        changed_any |= _write_or_check(
            interest_file.path,
            new_text=interest_text,
            check=args.check,
            show_diff=args.diff,
        )

        categories_text, category_ids = normalize_categories(categories_file)
        changed_any |= _write_or_check(
            categories_file.path,
            new_text=categories_text,
            check=args.check,
            show_diff=args.diff,
        )

        kinks_text = normalize_kinks(
            kinks_file,
            valid_category_ids=set(category_ids),
            duplicate_description_mode=args.duplicate_descriptions,
        )
        changed_any |= _write_or_check(
            kinks_file.path,
            new_text=kinks_text,
            check=args.check,
            show_diff=args.diff,
        )

    except (ValidationError, JsonSchemaValidationError) as exc:
        _die(str(exc))

    # Prettify css/app.css
    app_css = wwwroot / "css" / "app.css"
    if app_css.exists():
        new_css = prettify_css_text(app_css.read_text(encoding="utf-8"))
        changed_any |= _write_or_check(
            app_css,
            new_text=new_css,
            check=args.check,
            show_diff=args.diff,
        )
    else:
        _warn(f"Not found, skipping CSS prettify: {app_css}")

    if args.check and changed_any:
        print(
            "CHANGES NEEDED: formatting/ordering differs. Run without --check to apply."
        )
        return 2

    print(
        "OK: YAML validated/sorted and files formatted."
        if not changed_any
        else "OK: changes applied."
    )
    return 0


if __name__ == "__main__":
    main()
