#!/usr/bin/env python3
"""
Archive the ShareKinks project into wwwroot/sharekinks.zip.

- Safe to run from either:
  - project root (contains wwwroot/)
  - tools/ (a direct child of project root)
- Refuses to run if a project root with wwwroot/ cannot be determined.
- Excludes any files/dirs that start with "." (at any path segment), except for those in HIDDEN_PATHS_TO_INCLUDE.
- Excludes only the output zip file itself.
"""

from __future__ import annotations

import os
import sys
import zipfile
from pathlib import Path


ZIP_NAME = "sharekinks.zip"
WWWROOT_DIRNAME = "wwwroot"
TOOLS_DIRNAME = "tools"
ITEMS_TO_EXCLUDE = ["node_modules"]
HIDDEN_PATHS_TO_INCLUDE = [".vscode", ".markdownlint.yml"]


def find_project_root(start: Path) -> Path | None:
    """
    Determine the project root based on where the script is run from.
    """
    cwd = start.resolve()

    if (cwd / WWWROOT_DIRNAME).is_dir():
        return cwd

    if cwd.name == TOOLS_DIRNAME and (cwd.parent / WWWROOT_DIRNAME).is_dir():
        return cwd.parent

    return None


def is_exclude_path(path: Path, root: Path) -> bool:
    """
    True if any path segment is in the excluded items list.
    """
    try:
        rel = path.relative_to(root)
    except ValueError:
        return True

    return any(part in ITEMS_TO_EXCLUDE for part in rel.parts)

def is_hidden_path(path: Path, root: Path) -> bool:
    """
    True if any path segment (relative to root) starts with ".".
    """
    try:
        rel = path.relative_to(root)
    except ValueError:
        return True

    return any(part.startswith(".") for part in rel.parts)


def should_exclude(path: Path, *, root: Path, output_zip: Path) -> bool:
    if path.resolve() == output_zip.resolve():
        return True
    rel = path.relative_to(root)
    if rel.parts[0] in HIDDEN_PATHS_TO_INCLUDE:
        return False
    if is_exclude_path(path, root):
        return True
    if is_hidden_path(path, root):
        return True

    return False


def create_archive(project_root: Path) -> Path:
    wwwroot = project_root / WWWROOT_DIRNAME
    if not wwwroot.is_dir():
        raise RuntimeError(
            f"Refusing to run: '{project_root}' does not contain '{WWWROOT_DIRNAME}/'."
        )

    output_zip = wwwroot / ZIP_NAME

    with zipfile.ZipFile(output_zip, "w", zipfile.ZIP_DEFLATED) as zipf:
        for root_str, dirs, files in os.walk(project_root):
            root_path = Path(root_str)

            # Prevent descending into excluded directories
            dirs[:] = [
                d
                for d in dirs
                if not should_exclude(
                    root_path / d,
                    root=project_root,
                    output_zip=output_zip,
                )
            ]

            for filename in files:
                file_path = root_path / filename
                if should_exclude(
                    file_path,
                    root=project_root,
                    output_zip=output_zip,
                ):
                    continue

                arcname = file_path.relative_to(project_root)
                zipf.write(file_path, arcname)

    return output_zip


def main() -> int:
    project_root = find_project_root(Path.cwd())
    if project_root is None:
        print(
            f"Refusing to run: could not find a project root containing '{WWWROOT_DIRNAME}/'.\n"
            f"Run this script from the project root or from '{TOOLS_DIRNAME}/' under it.",
            file=sys.stderr,
        )
        return 2

    try:
        output_zip = create_archive(project_root)
    except RuntimeError as exc:
        print(str(exc), file=sys.stderr)
        return 2

    print(f"Created: {output_zip}")
    return 0


if __name__ == "__main__":
    raise SystemExit(main())
