/* eslint-disable no-console */

export interface Env {
  CORS_ALLOW_ORIGINS?: string; // comma-separated list
  SHAREKINKS_KV: KVNamespace;
  PUBLIC_BASE_URL: string; // optional, used if you want the API to emit full URLs
}

type Json = Record<string, unknown>;


function pickCorsOrigin(request: Request, env: Env): string | null {
  const origin = request.headers.get("Origin");
  if (!origin) {
    return null;
  }
  const allow = (env.CORS_ALLOW_ORIGINS || "").split(",").map((s) => s.trim()).filter(Boolean);
  if (allow.length === 0) {
    return "*";
  }
  if (allow.includes(origin)) {
    return origin;
  }
  return null;
}

function applyCors(headers: Headers, request: Request, env: Env): void {
  const picked = pickCorsOrigin(request, env);
  if (picked) {
    headers.set("Access-Control-Allow-Origin", picked);
    headers.set("Vary", "Origin");
    headers.set("Access-Control-Allow-Credentials", "false");
  }
  headers.set("Access-Control-Allow-Methods", "GET,POST,PUT,DELETE,OPTIONS");
  headers.set("Access-Control-Allow-Headers", "Content-Type, Authorization");
  headers.set("Access-Control-Max-Age", "86400");
}

type ShareRecordV1 = {
  v: 1;
  tokenHash: string; // base64url(sha256(token))
  createdAt: string; // ISO
  updatedAt: string; // ISO
  state: unknown; // JSON-serializable state blob from the frontend
};

function jsonResponse(
  request: Request,
  env: Env,
  data: unknown,
  init: ResponseInit = {},
  extraHeaders: HeadersInit = {},
): Response {
  const headers = new Headers(init.headers);
  headers.set("Content-Type", "application/json; charset=utf-8");

  // CORS
  applyCors(headers, request, env);

  headers.set("Cache-Control", "no-store");
  for (const [k, v] of Object.entries(extraHeaders)) {
    headers.set(k, v);
  }

  return new Response(JSON.stringify(data), { ...init, headers });
}

function textResponse(
  request: Request,
  env: Env,
  text: string,
  status = 200,
  extraHeaders: HeadersInit = {},
): Response {
  const headers = new Headers();
  headers.set("Content-Type", "text/plain; charset=utf-8");

  applyCors(headers, request, env);

  headers.set("Cache-Control", "no-store");
  for (const [k, v] of Object.entries(extraHeaders)) {
    headers.set(k, v);
  }

  return new Response(text, { status, headers });
}


function isOptions(req: Request): boolean {
  return req.method.toUpperCase() === "OPTIONS";
}

function base64UrlEncode(bytes: ArrayBuffer): string {
  const u8 = new Uint8Array(bytes);
  let s = "";
  for (const b of u8) s += String.fromCharCode(b);
  const b64 = btoa(s);
  return b64.replace(/\+/g, "-").replace(/\//g, "_").replace(/=+$/g, "");
}

function base64UrlEncodeU8(u8: Uint8Array): string {
  let s = "";
  for (const b of u8) s += String.fromCharCode(b);
  const b64 = btoa(s);
  return b64.replace(/\+/g, "-").replace(/\//g, "_").replace(/=+$/g, "");
}

function randomId(length = 10): string {
  // base64url over random bytes, then trim to desired length.
  const bytes = new Uint8Array(Math.ceil((length * 3) / 4) + 2);
  crypto.getRandomValues(bytes);
  return base64UrlEncodeU8(bytes).slice(0, length);
}

async function sha256Base64Url(input: string): Promise<string> {
  const data = new TextEncoder().encode(input);
  const digest = await crypto.subtle.digest("SHA-256", data);
  return base64UrlEncode(digest);
}

function timingSafeEqual(a: string, b: string): boolean {
  // Compare strings in constant time over their UTF-8 bytes.
  const aBytes = new TextEncoder().encode(a);
  const bBytes = new TextEncoder().encode(b);
  const len = Math.max(aBytes.length, bBytes.length);
  let diff = aBytes.length ^ bBytes.length;
  for (let i = 0; i < len; i++) {
    const av = i < aBytes.length ? aBytes[i] : 0;
    const bv = i < bBytes.length ? bBytes[i] : 0;
    diff |= av ^ bv;
  }
  return diff === 0;
}

function parseBearer(authHeader: string | null): string | null {
  if (!authHeader) return null;
  const m = authHeader.match(/^\s*Bearer\s+(.+?)\s*$/i);
  return m ? m[1] : null;
}

function kvKey(id: string): string {
  return `share:${id}`;
}

async function readShare(env: Env, id: string): Promise<ShareRecordV1 | null> {
  const raw = await env.SHAREKINKS_KV.get(kvKey(id));
  if (!raw) return null;
  try {
    const parsed = JSON.parse(raw) as ShareRecordV1;
    if (parsed && parsed.v === 1 && typeof parsed.tokenHash === "string") {
      return parsed;
    }
    return null;
  } catch {
    return null;
  }
}

async function writeShare(env: Env, id: string, rec: ShareRecordV1): Promise<void> {
  await env.SHAREKINKS_KV.put(kvKey(id), JSON.stringify(rec));
}

function routeMatch(pathname: string): { kind: "shareRoot" } | { kind: "shareId"; id: string } | null {
  // Supports:
  //   /api/share
  //   /api/share/<id>
  const parts = pathname.split("/").filter(Boolean);
  if (parts.length === 2 && parts[0] === "api" && parts[1] === "share") {
    return { kind: "shareRoot" };
  }
  if (parts.length === 3 && parts[0] === "api" && parts[1] === "share") {
    return { kind: "shareId", id: parts[2] };
  }
  return null;
}

async function parseJson(req: Request): Promise<Json> {
  const ct = req.headers.get("Content-Type") || "";
  if (!ct.toLowerCase().includes("application/json")) {
    throw new Error("Content-Type must be application/json");
  }
  return (await req.json()) as Json;
}

export default {
  async fetch(request: Request, env: Env): Promise<Response> {
    if (isOptions(request)) {
      // CORS preflight
      return jsonResponse(request, env, { ok: true });
    }

    const url = new URL(request.url);
    const route = routeMatch(url.pathname);
    if (!route) {
      return textResponse(request, env, "Not Found", 404);
    }

    if (route.kind === "shareRoot") {
      if (request.method.toUpperCase() !== "POST") {
        return textResponse(request, env,  "Method Not Allowed", 405);
      }

      let body: Json;
      try {
        body = await parseJson(request);
      } catch (err) {
        return jsonResponse(request, env, { error: String(err) }, { status: 400 });
      }

      const state = body.state;
      if (state === undefined) {
        return jsonResponse(request, env, { error: "Missing required field: state" }, { status: 400 });
      }
      
      const maxAttempts = 8;

       
      // Create a new share record.
      let id: string | null = null;

      for (let attempt = 0; attempt < maxAttempts; attempt++) {
        const candidate = randomId(10);
        const existing = await env.SHAREKINKS_KV.get(kvKey(candidate));
        if (existing === null) {
          id = candidate;
          break;
        }
      }

      if (!id) {
        return jsonResponse(
          request,
          env,
          { error: "Failed to allocate share ID (please retry)" },
          { status: 503 },
        );
      }
      const manageToken = randomId(32);
      const tokenHash = await sha256Base64Url(manageToken);
      const now = new Date().toISOString();

      const rec: ShareRecordV1 = {
        v: 1,
        tokenHash,
        createdAt: now,
        updatedAt: now,
        state,
      };

      await writeShare(env, id, rec);

      const base = env.PUBLIC_BASE_URL?.trim() || "";
      const publicUrl = base ? `${base}/?s=${encodeURIComponent(id)}` : undefined;
      const manageUrl = base ? `${base}/?m=${encodeURIComponent(id)}#${manageToken}` : undefined;

      return jsonResponse(request, env, {
        id,
        manageToken,
        publicUrl,
        manageUrl,
      });
    }

    // /api/share/:id
    const id = route.id;

    if (request.method.toUpperCase() === "GET") {
      const rec = await readShare(env, id);
      if (!rec) {
        return jsonResponse(request, env, { error: "Not found" }, { status: 404 });
      }
      return jsonResponse(request, env, { state: rec.state, updatedAt: rec.updatedAt, createdAt: rec.createdAt });
    }

    if (request.method.toUpperCase() === "PUT") {
      const authToken = parseBearer(request.headers.get("Authorization"));
      if (!authToken) {
        return jsonResponse(request, env, { error: "Missing Authorization: Bearer <token>" }, { status: 401 });
      }

      const rec = await readShare(env, id);
      if (!rec) {
        return jsonResponse(request, env, { error: "Not found" }, { status: 404 });
      }

      const providedHash = await sha256Base64Url(authToken);
      if (!timingSafeEqual(providedHash, rec.tokenHash)) {
        return jsonResponse(request, env, { error: "Unauthorized" }, { status: 401 });
      }

      let body: Json;
      try {
        body = await parseJson(request);
      } catch (err) {
        return jsonResponse(request, env, { error: String(err) }, { status: 400 });
      }

      const state = body.state;
      if (state === undefined) {
        return jsonResponse(request, env, { error: "Missing required field: state" }, { status: 400 });
      }

      const now = new Date().toISOString();
      const updated: ShareRecordV1 = { ...rec, state, updatedAt: now };
      await writeShare(env, id, updated);

      return jsonResponse(request, env, { ok: true, id, updatedAt: now });
    }

    if (request.method.toUpperCase() === "DELETE") {
      const authToken = parseBearer(request.headers.get("Authorization"));
      if (!authToken) {
        return jsonResponse(request, env, { error: "Missing Authorization: Bearer <token>" }, { status: 401 });
      }

      const rec = await readShare(env, id);
      if (!rec) {
        return jsonResponse(request, env, { error: "Not found" }, { status: 404 });
      }

      const providedHash = await sha256Base64Url(authToken);
      if (!timingSafeEqual(providedHash, rec.tokenHash)) {
        return jsonResponse(request, env, { error: "Unauthorized" }, { status: 401 });
      }

      await env.SHAREKINKS_KV.delete(kvKey(id));
      return jsonResponse(request, env, { ok: true, id });
    }

    return textResponse(request, env, "Method Not Allowed", 405);
  },
};
